//
// pit.C
//
// Version 2001/3/11 Jun Makino
// Created --- base structure stolen from flat_hermite.C (in Starlab)
// hermite.C
// hermite scheme with simple (flat-tree) hdyn structure
//#include "mpi.h"

#include "pit_system.h"
#include  <fstream.h>

#include <unistd.h>


void set_grape6_serialize_mode(int mode);

//=============================================================================
//  functions related to time steps
//=============================================================================

real adjust_number_to_power(real newstep, real max_step_size)
{
    real tmp = 1.0;
    real limit = max_step_size;
    if(newstep < limit) limit = newstep;
    while(tmp < limit) tmp *= 4.0;
    while(tmp > limit) tmp *= 0.5;
    return tmp;
}



//-----------------------------------------------------------------------------
// set_first_timestep -- calculate the first timestep using acc, jerk,
//                       and pot (acceleration, jerk, potential).
//                       The expression used here is:
//                                                 ________
//               1              /   | acc |       V | pot |   \
//   timestep = ---  eta * min (   ---------  ,  -----------   )
//               32             \   | jerk |       | acc |    /
//
//  The first expression gives the time scale on which the acceleration is
//  changing significantly.
//  The second expression gives an estimate for the free fall time
//  in the case of near-zero initial velocity.
//-----------------------------------------------------------------------------

void particle::set_first_timestep(real eta,
				  real step_limit)
{
    real eta_init = eta / 32;	// initial accuracy parameter
    real j2 = jerk * jerk;
    real a2 = acc * acc;
    real dt_adot = eta_init/4;	// time step based on rate of change of acc
    real dt_ff   = eta_init/4;	// time step based on free-fall time
    real dt;			// minimum of the two

    if (j2 > 0)
	dt_adot = eta_init * sqrt(a2 / j2);

    if (pot < 0 && a2 > 0)
	dt_ff = eta_init * sqrt(-pot / a2);

    dt = min(dt_adot, dt_ff);

    // Apply an upper limit to the time step:

    dt = min(dt, eta_init/4);	// eta_init/4 is arbitrary limit...

    real true_limit = step_limit;
    if (dt < 1e-7){
	cerr << "Short step for " << index << " " << pos << " " <<vel << "\n       " <<acc << " " <<jerk <<endl;
    }

    timestep = adjust_number_to_power(dt, true_limit);

    if (time != 0)
	while (fmod(time, timestep) != 0)
	    timestep *= 0.5;
    
    real tnext = time + timestep;

    if (timestep == 0 || time == tnext) {

	cerr << endl << "set_first_timestep: dt = 0 at time "
	     << time << endl;
	exit(0);
    }
}

//-----------------------------------------------------------------------------
// new_timestep:  Calculate the next timestep following the Aarseth
//                formula (Aarseth 1985), including necessary adjustments
//                for the hierarchical timestep scheme.
//-----------------------------------------------------------------------------


local inline real new_timestep(vector& at3,		// 3rd order term
			       vector& bt2,		// 2nd order term
			       vector& jerk,		// 1st order term
			       vector& acc,		// 0th order term
			       real timestep,		// old timestep
			       real time,		// present time
			       real eta,
			       real max_step_size)
{
    if (timestep == 0)
	cerr << "new_timestep: old timestep = 0" << endl;

    // N.B. "timestep" may really be dtau (if b->get_kappa() > 1).

    real newstep, altstep;

    real dist, dtff2, dtv2;
    real a2, j2, k2, l2;
    real tmp1, tmp2, tmp3, tmp4;

    // Use the Aarseth criterion here.
    
    real dt2 = timestep * timestep;
    real dt4 = dt2 * dt2;
    real dt6 = dt4 * dt2;
    
    // A note on terminology: these scaled quantities are simply the
    // derivatives of the acceleration -- a, j = a', k = a'', l = a'''.
    // Old notation called j2 a1scaled2, k2 a2scaled2, and l2 a3scaled2.
    // I prefer the new names... (Steve, 9/99)

    l2 = 36 * at3 * at3 / dt6;			// = (a''')^2
    k2 = 4 * bt2 * bt2 / dt4;			// = (a'')^2
    j2 = jerk * jerk;				// = (a')^2
    a2 = acc * acc;
    
    real aarsethstep = 0;
    
    // Break the calculation up into pieces, for debugging purposes.
    
    // Numerator:
    
    real sqrt1 = 0;
    tmp1 = a2 * k2;
    if (tmp1 > 0) sqrt1 = sqrt(tmp1);
    tmp2 = sqrt1 + j2;
    
    // Denominator:
    
    real sqrt3 = 0;
    tmp3 = j2 * l2;
    if (tmp3 > 0) sqrt3 = sqrt(tmp3);
    tmp4 = sqrt3 + k2;
    
    if (tmp2 > 0 && tmp4 > 0)
	aarsethstep = eta * sqrt(tmp2/tmp4);
    
    // Stop and flag an error if any of the intermediates
    // exceeds VERY_LARGE_NUMBER.
    
    if (tmp1 >= 0 && tmp2 >= 0 && tmp3 >= 0 && tmp4 >= 0
	&& tmp1 < VERY_LARGE_NUMBER
	&& tmp2 < VERY_LARGE_NUMBER
	&& tmp3 < VERY_LARGE_NUMBER
	&& tmp4 < VERY_LARGE_NUMBER) {	// test negation to catch NaN,
	// which always tests false
    } else {
	
	cerr.precision(HIGH_PRECISION);
	cerr << endl << "new_timestep:  found errors at time "
	     << time << endl;
	
	PRL(acc);
	PRL(jerk);
	PRL(bt2);
	PRL(at3);
	PRC(abs(acc)); PRL(abs(jerk));
	PRC(a2); PRL(j2);
	PRC(abs(bt2)); PRL(abs(at3));
	PRC(k2); PRL(l2);
	PRC(tmp1); PRC(tmp2); PRC(tmp3); PRL(tmp4);
	

		cerr << endl;
    }
    
    newstep = aarsethstep ;

    // The natural time step is newstep.  Force it into an appropriate
    // power-of-two block.  A halving of timestep is always OK.  To
    // preserve the synchronization, a doubling is OK only if the
    // current time could be reached by an integral number of doubled
    // time steps (see use of fmod below).



    if (newstep < timestep)

	return 0.5 * timestep;

    else if (newstep < 2 * timestep)

	return timestep;

    else if (fmod(time, 2 * timestep ) != 0.0
	     || 2 * timestep > max_step_size)

	return timestep;

    else {
	return 2.0*timestep;
    }
}

void particle_system::predict_loworder_all(real time)
{
    int i;
    particle * b;
    for(i=0,b=pb;i<n;i++,b++){
	b->predict_loworder(time);
    }
}


//-----------------------------------------------------------------------------
// correct_and_update:  Apply the corrector for 3rd-order hermite scheme
//			and update time and timestep.
//
// See Makino & Aarseth (PASJ 1992) for derivation of corrector terms.
//
//-----------------------------------------------------------------------------

local inline void get_derivs(vector& acc, vector& jerk,
			     vector& old_acc, vector& old_jerk,
			     real dt, vector& bt2, vector& at3)
{
    bt2 = -3 * (old_acc - acc) - dt * (2 * old_jerk + jerk);
    at3 =  2 * (old_acc - acc) + dt * (old_jerk + jerk);
}
void particle::update(vector& bt2, vector& at3,
		      real eta,real max_step_size) 
{
    time += timestep;
    real dt = timestep;

    // Reminder:	at3  =  a''' dt^3 / 6		(a' = j)
    //			bt2  =  a''  dt^2 / 2


    // Define correction factor for use in new_timestep.


    real new_dt = new_timestep(at3, bt2, jerk, acc, dt, time, eta, max_step_size);


    // Computed new_dt will be timestep if slow not set; dtau otherwise.

    timestep = new_dt;
}

void particle::correct_and_update(real eta, real max_step_size)
{

    real dt = timestep;

    vector bt2, at3;

    //    cerr < "new_timestep for "; PRL(index); write(cerr);

    get_derivs(acc, jerk, old_acc, old_jerk, dt, bt2, at3);

    // Reminder:	bt2  =  a''  dt^2 / 2		(a' = j)
    //			at3  =  a''' dt^3 / 6


    vector new_pos = pred_pos + dt*dt*(0.05 * at3 + ONE12 * bt2);
    vector new_vel = pred_vel + dt*(0.25 * at3 + ONE3 * bt2) ;
    // PRC(index); PRL(new_pos);
    // PRC(time);PRC(timestep);PRL(new_vel);
    //    PRC(new_pos*new_pos);PRC(new_pos*new_vel);PRL(new_vel*new_vel);
    // PRC(old_acc);    PRL(old_jerk);
    // PRC(acc);    PRL(jerk);
    pos = new_pos;
    vel = new_vel;

    //--------------------------------------------------------
    // Call update here to avoid recomputation of bt2 and at3.
    //--------------------------------------------------------

    update(bt2, at3, eta, max_step_size);

    return;
}


int particle_system::integrate_list(int n_next)

{
    static bool restart_grape = false;
    int i, steps = 0;
    real sys_t = time;
    real t_next = nt[0].pptr->get_next_time();
    nupdated = 0;
    calculate_acc_and_jerk_for_list(n_next,restart_grape);

    // Apply corrector and redetermine timesteps.
    
    bool reinitialize = false;
    
    bool diag = false;
    for (i = nupdated; i < n_next; i++) {
	particle *bi = nt[i].pptr;
	bi->correct_and_update(eta,max_timestep);
	bi->init_pred();
	bi->store_old_force();
    }

    update_grape_data(n_next);
    
    steps += n_next;
    return steps;
}


// scheduling stuff...

local inline void swap(node_time a[], int i, int j)
{
    // ASSUME assignment a = b is OK for structures.

    node_time tmp;

    tmp = a[i];
    a[i] = a[j];
    a[j] = tmp;
}

// Makino's quicksort -- recursive.

local void quicksort(node_time a[], int left, int right)
{
    // Sort nodes in array a, according to get_next_time().

    int i,j;
    real v;

    if (right > left) {
	bool is_sorted = true;
	for (i=left; i< right; i++) {
	    if (a[i].next_time > a[i+1].next_time) {
		is_sorted = false;
		i = right;
	    }
	}
	if (is_sorted) return;
	i = (left + right)/2;
	swap(a, i, right);
	v = a[right].next_time; i = left- 1; j = right;
	for (;;) {
	    while (a[++i].next_time < v && i < right) ;
	    while (a[--j].next_time > v && j > left) ;
	    if (i >= j) break;
	    swap(a, i, j);
	}
	swap(a, i, right);
	quicksort(a, left, i - 1);
	quicksort(a, i + 1, right);
    }
}


// Quick and dirty merging of an array consisting of two ordered
// subarrays.  2-3 times faster than Makino's quicksort.  (Steve 5/99)

inline local void QS(node_time a[], int n)
{
    quicksort(a, 0, n-1);
}

local void sort_node_array(node_time nodes[], int nprev)
{


    // Update times in the node_time structure.
    //    PRL(nprev);

    for (int i = 0; i < nprev; i++)
	nodes[i].next_time = nodes[i].pptr->get_next_time();

    if (nprev > 1) {
	QS(nodes, nprev);
    }
    //    PRL(nodes[0].next_time);
    //    PRL(nodes[0].pptr->get_next_time());
}

void check_and_initialize_node_array(node_time* &nt,
				     int & nprev,
				     int ntsize,
				     particle* pb,
				     int nbody,
				     int nsize)
{
    //    PRL(nprev); PRL(ntsize);
    //    PRL(nbody); PRL(nsize);
    if (ntsize < nsize) {
	ntsize = nsize;
	if (nt) delete [] nt;
	nt = new node_time[ntsize];
    }
    
    int i;
    particle* p;
    for(i=0,p=pb; i<nbody; i++,p++){
	nt[i].pptr = p;
	nt[i].next_time = p->get_next_time();
    }
    nprev=nbody;
}    

void rearrange_particle_array(node_time* &nt,
				     particle* pb,
				     int nbody)
{
    //    PRL(nprev); PRL(ntsize);
    //    PRL(nbody); PRL(nsize);
    static int invpsize = 0;
    static int * invp = NULL;
    if (invpsize < nbody) {
	invpsize = nbody;
	if (invp) delete [] invp;
	invp = new int[invpsize];
    }
    
    int i;
    for(i=0; i<nbody; i++){
	int loc = nt[i].pptr - pb;
	//	PRC(i); PRC(loc); PRL(nt[i].pptr->get_index());
	invp[loc] = i;
    }
    particle p;
    particle* pp;
    for(i=0; i<nbody; i++){
	if (invp[i] != i){
	    int j = invp[i];
	    //	    PRC(i); PRL(j);
	    p = *(nt[i].pptr);
	    pp = nt[i].pptr;
	    *(nt[i].pptr) = *(nt[j].pptr);
	    *(nt[j].pptr) = p;
	    nt[i].pptr = nt[j].pptr;
	    nt[j].pptr = pp;
	    invp[i] = i;
	    invp[nt[j].pptr - pb]=j;
	}
    }

    // consistency check...
    for(i=0; i<nbody; i++){
	if ((nt[i].pptr - pb) != i){
	    cerr << "rearrange failed..."<<endl;
	    PRC(i); PRL(nt[i].pptr - pb);
	}
	//	PRC(i); PRC(nt[i].next_time); PRC(nt[i].pptr->get_timestep());PRL(nt[i].pptr->get_index());
    }
}

void particle_system::fast_get_nodes_to_move(int &nlist,
					     real & tmin,
					     bool & reset)
{
    if (reset) {
	check_and_initialize_node_array(nt, nprev,nsize, pb, n, nsize);
    }
    reset = false;

    bool debug = false;
    // debug = (b->get_system_time() > 100);

    sort_node_array(nt, nprev);

    nlist = 0;
    tmin = nt[0].next_time;
    for (int i = 0; i < n; i++) {
	if (nt[i].next_time < tmin) {

	    // Fatal scheduling error...

	    cerr << endl << "Error in fast_get_nt_to_move():" << endl;
	    cerr.precision(HIGH_PRECISION);

	    PRC(tmin), PRL((real)tmin);
	    PRC(i); cerr << "node " << nt[i].pptr->get_index() << endl;
	    cerr << "time " << nt[i].pptr->get_time() << endl;
	    cerr << "timestep " << nt[i].pptr->get_timestep() << endl;
	    PRL(nt[i].next_time);
	    exit(-1);

	} else if (nt[i].next_time > tmin) {

	    i = n;	// aka break!

	} else {
	    nlist = i + 1;
	}
    }
    nprev = nlist;
}



// initialize_system_phase1:  Set current time and "predicted" position
//                            and velocity for all nodes.
void particle_system::initialize_system_phase1(real t)
{
    int i;
    particle* b;  
    for (b = pb,i=0; i<n; b++,i++) {
	b->set_time(t);
	b->init_pred();
    }
}

void set_some_value_in_place_of_zero_jerk(particle* pb,int n)
{
    int i;
    particle* b;  
    for (b = pb,i=0; i<n; b++,i++) {
	if (square(b->get_jerk()) < 1e-200)b->set_jerk(vector(1e-20,0,0));
    }
}
 

// initialize_system_phase2:  Calculate acc, jerk, timestep, etc for all nodes.
// System should be synchronized prior to calling this function.

void particle_system::initialize_system_phase2(int call_id,
					       bool set_dt)	
{
    bool restart_grape = false;
    check_and_initialize_node_array(nt, nprev,ntsize, pb, n, nsize);
    for(int i = 0;i<6; i++){
	if (myprocid == 0)
	    cerr << "initialize --- " << i << "th try\n";
	nupdated = n;
	set_some_value_in_place_of_zero_jerk(pb, n);
	calculate_acc_and_jerk_for_list(n, restart_grape);
	update_grape_data(n);
    }
    real min_dt = VERY_LARGE_NUMBER;
    int i;
    particle* b;
    for(i=0,b=pb;i<n;i++,b++){
	b->store_old_force();
	if (set_dt|| b->get_timestep() <= 0)  {
	    b->set_first_timestep(eta,max_timestep); 
	    min_dt = min(min_dt, b->get_timestep());
	}
    }
}

void particle_system::full_reinitialize(real t, bool verbose)
{
    cerr << "\nReinitializing system at time " << t << endl;

    real cpu_0 = cpu_time();

    set_time(t);
    initialize_system_phase1(t);
    initialize_system_phase2(3,true);
    update_grape_data(n);
    if (verbose && (myprocid == 0)) {
	cerr << "CPU time for reinitialization = "
	     << cpu_time() - cpu_0 << endl;
    }
}

// bh handling

void particle_system::initialize_bh_storage()
{
    bhp = new particle[nbh];
    bhlocs = new int[nbh];
    for(int i = 0;i<n; i++){
	if (pb[i].get_index() < nbh){
	    pb[i].set_isbh(1);
	}
    }
    setup_bh_indices();
}
void particle_system::setup_bh_indices()
{
    int ibh = 0;
    for(int i = 0;i<n; i++){
	if (pb[i].get_isbh()){
	    bhlocs[ibh] = i;
	    ibh++;
	}
    }
    if (ibh > 0){
	cerr << "Myid = " << myprocid << " nbh = " << ibh<<endl;
	for(int i = 0;i<ibh;i++){
	    cerr << "bh[" <<i  << "]= " << bhlocs[i]<<endl;
	}
    }
	    
		
}
void particle_system::collect_BH_data()
{
    //    cerr << "collect_BH_data called "<< nbh << endl;
    //    for(int i=0;i<nbh; i++)cerr << bhlocs[i] <<endl;
    if (myprocid == 0){
	for(int i = 0;i<nbh;i++){
	    bhp[i] = pb[bhlocs[i]];
	}
    }
    //    cerr << "collect_BH_data end "<< nbh << endl;
}

// energy calculation etc.



void particle_system::compute_energies(real & epot, real & ekin, real & etot)
{
    epot = ekin = etot = 0;
    int i;
    particle* b;
    for(i=0,b=pb;i<n;i++,b++){	
	real mi = b->get_mass();
	epot += 0.5 * mi * b->get_pot();
	vector vel = b->get_vel();
	ekin += 0.5 * mi * vel * vel;
    }
    etot = ekin + epot;
}

void particle_system::print_energies(int mode, ostream& s )
{
    real epot;
    real ekin;
    real etot;
    static real old_etot;
    compute_energies(epot, ekin, etot);
    MP_collect_energies(epot, ekin, etot);

    if (myprocid == 0){
	s << "Energies(KE,PE,ETOT)=  " << ekin << "  " << epot << "  " << etot;
	if (mode) {
	    real de = (etot - old_etot) / etot;
	    s << ";  de = " << de << "\n";
	} else {
	    old_etot = etot;
	    s << "\n";
	}
    }
}

void particle_system::print_integration_stats(ostream& s)
{
    MP_step_count = step_count;
    MP_collect_integration_stats();
    if (myprocid == 0){
	s << "steps = " <<MP_step_count
	  << " Bsteps = " <<blockstep_count
	  << " CPU time " << cpu_time() << endl;
    }
    MP_print_integration_stats(s);

}

void particle_system::print_cmterms(ostream& s)
{
    pos = 0.0;
    vel = 0.0;
    mass = 0.0;
    particle * p = pb;
    for(int i = 0; i<n;i++){
	pos += p->get_mass()*p->get_pos();
	vel += p->get_mass()*p->get_vel();
	mass += p->get_mass();
	p++;
    }
    MP_collect_cmterms(pos,vel,mass);
    if(myprocid == 0){
	pos/= mass;
	vel/= mass;
	s << "CMPOS= " << pos <<endl;
	s << "CMV = "  << vel <<endl;
    }
}

void particle_system::print_log(int mode, ostream& s)
{
    if(myprocid == 0) s << "Time= " << time << " N= " << n*nprocessors <<endl;
    print_energies(mode, s);
    print_cmterms(s);
    print_BH(s);
    print_integration_stats(s);
    s << endl;
}

void particle_system::print_BH( ostream& s)
{
    if(myprocid == 0) {
	for(int i=0;i<nbh; i++){
	    particle* bp = pb+bhlocs[i];
	    s << "BH" << i << " " << bp->get_pos()
	      << " " << bp->get_vel()
	      << " " << bp->get_pot() <<endl;
	}
    }
}
void particle_system::print_initial_log(ostream& s)
{
    if(myprocid == 0) {
	s << "n = "   << n  <<endl;
	s << "nbh = " <<nbh <<endl;
	s << "t_end = " << t_end << endl;
	s << "eta = " << eta << endl;
	s << "eps2 = " << eps2 << endl;
	s << "dt_snap_next = " << dt_snap_next << endl;
	s << "dt_log_next = " << dt_log_next << endl;
	s << "max_timestep = " << max_timestep << endl;
    }
}

void particle_system::check_time_and_output_log(real tnext, ostream& s)
{
    if (tnext > t_log_next){
	print_log(1,s);
	t_log_next += dt_log_next;
    }
}
    
void particle_system::check_time_and_output_snap(real tnext, ostream& s)
{
    if ((tnext > t_snap_next)){
	if (snapflag){
	    cerr << "creating output snap... \n";
	    write(s);
	}
	t_snap_next += dt_snap_next;
#ifdef GRAPHICS
	cerr << "Plot particles at " << time<<endl;
	plot_particles(n, pb, time);
#endif	    
    }
}
    


void particle_system::evolve_system(ostream & snapout,
				    ostream & logout,
				    ostream & dumpout)
{
    // Initialization:
    // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    // Convenient for now to pass request for full dump and also the
    // dump interval via dt_snap...
    logout.precision(HIGH_PRECISION);

    initialize_bh_storage();
#ifdef GRAPHICS
	initialize_plot(0.5);
#endif	    

    while ( (max_timestep > dt_snap_next) || (max_timestep > dt_log_next)) max_timestep /= 2.0;
    
    full_reinitialize(time, true);


    print_initial_log(logout);
    print_log(0, logout);
    bool reset_nodes = true;
    long diagsteps = 0;
    while (time <= t_end) {
	int n_next;
	real ttmp;
	fast_get_nodes_to_move(n_next, ttmp, reset_nodes);

	//
	// The next function works fine with g++, but not with cxx...
	//
	//	if(n_next == n) {
	//         rearrange_particle_array(nt, pb, n);
	//         setup_bh_indices();
	//      }
	MP_adjust_current_time_and_nnext(n_next, ttmp);
	if (step_count > diagsteps ){
	    if (myprocid == 0)cerr << "Time = " << ttmp  <<  " nodes="  << n_next << " steps " << step_count <<endl;
	    diagsteps += 500000;
	}
	
	check_time_and_output_log(ttmp,logout);
	check_time_and_output_snap(ttmp, snapout);
	
	if (ttmp > t_end) return;
	
	real t_prev = time;
	
	set_time(ttmp);
	// Take a new step to time t (now system_time):
	
	int ds = integrate_list(n_next);
	
	step_count += ds;
	grape_steps += ds;
	blockstep_count += 1;
    }
    close_grape();
}

void print_usage_and_exit()
{
    cerr <<
	 "usage: pit -t # -a # -d # -D # -e # -N #"
	 << "[-c \"...\"]      "
	 << "for t (time span),\n a (accuracy parameter ), "
	 << "d (output interval), D (snapshot output interval), "
	 << "b (number of bh p ), l (log file name), "
	 << "o (snapout file name ), u (dump file name), "
	 << "e (softening length), \n";
    exit(1);
}



main(int argc, char **argv)
{
    particle_system s;			// particle root node

    real delta_t = 10;		// time span of the integration
    real dtout = .25;		// output interval--make a power of 0.5
    real dt_snap;		// snap output interval
    real eps = 0.01;		// softening length               
    real eta = 0.1;		// time step parameter
    int npoints = 0;            // number of point-mass particles
    int grape_nclusters = 1;
    int grape_firstcluster = 0;
    char *comment;		// comment string
    char ifname[255]; 
    ifstream fsnapin;
    char ofname[255];
    ofstream fsnapout;
    char logfname[255];
    ofstream flogout;
    char dumpfname[255];
    ofstream fdumpout;
    extern char *poptarg;
    int nbh = 0; //number of point-mass massive particles
    int pgetopt(int, char **, char *), c;

    bool a_flag = false;
    bool b_flag = false;
    bool c_flag = false;
    bool d_flag = false;
    bool D_flag = false;
    bool e_flag = false;
    bool i_flag = false;
    bool l_flag = false;
    bool q_flag = false;
    bool o_flag = false;
    bool t_flag = false;
    bool u_flag = false;
    bool N_flag = false;

    // rough structure of the code
    //
    // parse parameters
    // read input snapshot
    // initialize integrator
    // integrator main loop
    // postprocessing

    wall_init();
    s.MP_initialize(argc,argv);
    
    if (s.get_myprocid()==0){
	while ((c = pgetopt(argc, argv, "a:b:c:d:D:e:f:g:i:l:N:o:qt:")) != -1) {
	    switch (c) {
		case 'a':
		    a_flag = true;
		    eta = atof(poptarg);
		    break;
		case 'b':
		    b_flag = true;
		    nbh = atoi(poptarg);
		    break;
		case 'c':
		    c_flag = true;
		    comment = poptarg;
		    break;
		case 'd':
		    d_flag = true;
		    dtout = atof(poptarg);
		    break;
		case 'D':
		    D_flag = true;
		    dt_snap = atof(poptarg);
		    break;
		case 'e':
		    e_flag = true;
		    eps = atof(poptarg);
		    break;
		case 'f':
		    grape_firstcluster = atoi(poptarg);
		    break;
		case 'g':
		    grape_nclusters = atoi(poptarg);
		    break;
		case 'i': strcpy(ifname,poptarg);
		    i_flag = 1;
		    break;
		case 'l': strcpy(logfname,poptarg);
		    l_flag = 1;
		    break;
		case 'o': strcpy(ofname,poptarg);
		    o_flag = 1;
		    break;
		case 'u': strcpy(dumpfname,poptarg);
		    u_flag = 1;
		    break;
		case 'q':
		    q_flag = true;
		    break;
		case 't':
		    t_flag = true;
		    delta_t = atof(poptarg);
		    break;
		case 'N':
		    N_flag = true;
		    npoints = atoi(poptarg);
		    break;
		case '?':
		    print_usage_and_exit();
	    }
	}
	cerr << "Nclusters = " << grape_nclusters <<endl;
	cerr << "Fcluster = " << grape_firstcluster <<endl;
	if (!q_flag) {
	    // Check input arguments and echo defaults.
	    
	    if (!t_flag)
		cerr << "default delta_t = " << delta_t << "\n";
	    if (!d_flag)
		cerr << "default dtout = " << dtout << "\n";
	    if (!a_flag)
		cerr << "default eta = " << eta << "\n";
	    if (!e_flag)
		cerr << "default eps = " << eps << "\n";
	}
    }
    MP_convert_snap_name(i_flag,ifname);
    if (i_flag == false) {cerr << s.get_myprocid() <<" -i fname required\n"; exit(0);}
    MP_convert_snap_name(o_flag,ofname);
    fsnapin.open(ifname,ios::in);
    s.read(fsnapin);
    if (o_flag){
	s.set_snapflag(1);
	fsnapout.open(ofname,ios::out);
    }
    if (l_flag){
	if (s.get_myprocid() == 0)flogout.open(logfname,ios::out);
    }
    if (!D_flag)dt_snap = delta_t;
    MP_copyparams(eta,eps,dtout,dt_snap,delta_t,nbh,grape_nclusters,grape_firstcluster);
    s.set_nbh(nbh);
    s.set_eta(eta);
    s.set_eps2(eps*eps);
    s.set_dt_log_next(dtout);
    s.set_dt_snap_next(dt_snap);
    s.set_t_end(s.get_time()+delta_t);
    s.set_t_log_next(s.get_time()+dtout);
    s.set_t_snap_next(s.get_time()+dt_snap);
    s.set_grape_nclusters(grape_nclusters);
    s.set_grape_firstcluster(grape_firstcluster);
    cerr << "Nclusters = " << grape_nclusters << " " << s.get_grape_nclusters() <<endl;  
    cerr.precision(HIGH_PRECISION);
    cout.precision(HIGH_PRECISION);
    if (l_flag){
	s.evolve_system(fsnapout,flogout,fdumpout);
    }else{
	s.evolve_system(fsnapout,cout,fdumpout);
    }
    MP_end();
    exit(0);
}

